<?php
$apiBaseUrl = 'http://localhost/jps-service/backend/routers/';
$clientesUrl = $apiBaseUrl . 'cliente.php';
$servicosTipoUrl = $apiBaseUrl . 'servico_tipo.php';
$servicosUrl = $apiBaseUrl . 'servico.php';
$atendentesUrl = $apiBaseUrl . 'servico_especialistas.php';
$chamadosUrl = $apiBaseUrl . 'chamados.php';
$contratosUrl = $apiBaseUrl . 'contrato.php';
$contratoMaquinasUrl = $apiBaseUrl . 'contrato_maquina.php';
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sistema de Abertura de Chamados</title>
    <style>
        .required-field::after {
            content: " *";
            color: red;
        }
        .loading {
            display: none;
            text-align: center;
            margin: 20px 0;
        }
    </style>
</head>
<body>
    <div class="container-fluid mt-4">
            <div class="col-lg-10">
                <div class="card">
                    <div class="card-header bg-primary text-white">
                        <h3 class="mb-0">Abertura de Chamado</h3>
                    </div>
                    <div class="card-body">
                        <form id="chamadoForm">
                            <p>
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label for="maquina" class="form-label required-field">Máquina/Equipamento</label>
                                    <div class="input-with-icon">
                                        <input type="text" class="form-control" id="maquina" name="maquina" required>
                                        <i class="fas fa-search search-icon" id="btnAbrirBuscaMaquina"></i>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <label for="fk_contrato" class="form-label">Contrato</label>
                                    <input type="text" class="form-control" id="fk_contrato" name="fk_contrato" required readonly>
                                </div>
                            </div>

                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label for="fk_cliente" class="form-label required-field">Cliente</label>
                                    <select class="form-control" id="fk_cliente" name="fk_cliente" required disabled>
                                        <option value="">Selecione um cliente</option>
                                    </select>
                                </div>
                                <div class="col-md-6">
                                    <label for="fk_cliente" class="form-label required-field">Solicitante</label>
                                    <input type="text" class="form-control" id="cliente_solicitante" name="cliente_solicitante">
                                </div>
                            </div>


                            <div class="row mb-3">
                                <div class="col-md-4">
                                    <label for="serie" class="form-label required-field">Número de Série</label>
                                    <input type="text" class="form-control" id="serie" name="serie" required>
                                </div>
                                <div class="col-md-4">
                                    <label for="modelo" class="form-label required-field">Modelo</label>
                                    <input type="text" class="form-control" id="modelo" name="modelo" required>
                                </div>
                                <div class="col-md-4">
                                    <label for="setor" class="form-label required-field">Setor</label>
                                    <input type="text" class="form-control" id="setor" name="setor" required>
                                </div>
                            </div>

                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label for="endereco" class="form-label required-field">Endereço</label>
                                    <input type="text" class="form-control" id="endereco" name="endereco" required>
                                </div>
                                <div class="col-md-4">
                                    <label for="cidade" class="form-label required-field">Cidade</label>
                                    <input type="text" class="form-control" id="cidade" name="cidade" required>
                                </div>
                                <div class="col-md-2">
                                    <label for="uf" class="form-label required-field">UF</label>                                    
                                    <input type="text" class="form-control" id="uf" name="uf" required>
                                </div>
                            </div>

                            <div class="row mb-3">
                                <div class="col-md-4">
                                    <label for="fk_servico_tipo" class="form-label required-field">Tipo de Serviço</label>
                                    <select class="form-select" id="fk_servico_tipo" name="fk_servico_tipo" required>
                                        <option value="">Selecione um tipo</option>
                                        <!-- Opções serão carregadas via AJAX -->
                                    </select>
                                </div>
                                <div class="col-md-4">
                                    <label for="fk_servico" class="form-label required-field">Serviço</label>
                                    <select class="form-select" id="fk_servico" name="fk_servico" required disabled>
                                        <option value="">Selecione o tipo primeiro</option>
                                    </select>
                                </div>
                                <div class="col-md-4">
                                    <label for="fk_atendente" class="form-label required-field">Atendente</label>
                                    <select class="form-select" id="fk_atendente" name="fk_atendente" required disabled>
                                        <option value="">Selecione o serviço primeiro</option>
                                    </select>
                                </div>
                            </div>

                            <div class="row mb-3">
                                <div class="col-md-12">
                                    <label for="detalhes" class="form-label required-field">Detalhes do Problema</label>
                                    <textarea class="form-control" id="detalhes" name="detalhes" rows="4" required></textarea>
                                </div>
                            </div>

                            <div class="row mb-3">
                                <div class="col-md-12">
                                    <label for="anexo" class="form-label">Anexo (opcional)</label>
                                    <input type="file" class="form-control" id="anexo" name="anexo">
                                    <small class="text-muted">Formatos aceitos: JPG, PNG, PDF (até 2MB)</small>
                                </div>
                            </div>

                            <div class="d-grid gap-2 d-md-flex justify-content-md-end">
                                <button type="button" class="btn btn-secondary me-md-2" onclick="limparFormulario()">Limpar</button>
                                <button type="submit" class="btn btn-primary">Abrir Chamado</button>
                            </div>
                        </form>

                        <div id="loading" class="loading">
                            <div class="spinner-border text-primary" role="status">
                                <span class="visually-hidden">Carregando...</span>
                            </div>
                            <p>Processando seu chamado...</p>
                        </div>

                        <div id="feedback" class="mt-3"></div>
                    </div>
                </div>
            </div>
    </div>

    <!-- Modal de Busca -->
    <div class="modal fade" id="modalBuscaMaquina" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-xl" style="max-width: 1000px;">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Buscar Máquina/Equipamento</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="row mb-3">
                        <div class="col-md-8">
                            <input type="text" id="inputBuscaMaquina" class="form-control" placeholder="Digite em Minúsculo para pesquisar...">
                        </div>
                        <div class="col-md-4">
                            <button id="btnBuscarMaquinas" class="btn btn-primary">Buscar</button>
                        </div>
                    </div>
                    <div class="table-responsive">
                        <table class="table table-striped table-hover" id="tabelaMaquinas">
                            <thead>
                                <tr>
                                    <th>Contrato</th>
                                    <th>Matriz</th>
                                    <th>Equipamento</th>
                                    <th>Modelo</th>
                                    <th>Série</th>
                                    <th>Setor</th>
                                    <th>Cidade</th>
                                    <th>UF</th>
                                    <th>Ação</th>
                                </tr>
                            </thead>
                            <tbody id="corpoTabelaMaquinas">
                                <!-- Dados serão carregados via AJAX -->
                            </tbody>
                        </table>
                    </div>
                    <div class="row">
                        <div class="col-md-6">
                            <div id="infoPaginacao" class="mt-2">Mostrando 0 de 0 registros</div>
                        </div>
                        <div class="col-md-6 text-end">
                            <nav>
                                <ul class="pagination justify-content-end" id="paginacaoMaquinas">
                                    <!-- Paginação será gerada dinamicamente -->
                                </ul>
                            </nav>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Adicione este estilo no <head> -->
    <style>
        .search-icon {
            position: absolute;
            right: 10px;
            top: 50%;
            transform: translateY(-50%);
            cursor: pointer;
            z-index: 10;
        }
        .input-with-icon {
            position: relative;
        }
        #tabelaMaquinas tbody tr {
            cursor: pointer;
        }
        #tabelaMaquinas tbody tr:hover {
            background-color: #f5f5f5;
        }
        /* Adicione isso no seu bloco de estilos existente */
        #tabelaMaquinas {
            font-size: 12px;
        }
        #tabelaMaquinas th, #tabelaMaquinas td {
            padding: 0.3rem; /* Reduz o padding para economizar espaço */
        }
        /* Estilo para a paginação */
        .pagination {
            flex-wrap: wrap;
            margin: 0;
        }

        .page-item {
            margin: 2px;
        }

        .page-link {
            padding: 0.25rem 0.5rem;
            font-size: 12px;
            min-width: 30px;
            text-align: center;
        }

        .page-item.disabled .page-link {
            opacity: 0.5;
        }

        .page-item.active .page-link {
            background-color: #0d6efd;
            border-color: #0d6efd;
        }
    </style>

    <script>
        // URLs da API
        const API_BASE_URL = 'http://localhost/jps-service/backend/routers/';
        const CLIENTES_URL = API_BASE_URL + 'cliente.php';
        const SERVICOS_TIPO_URL = API_BASE_URL + 'servico_tipo.php';
        const SERVICOS_URL = API_BASE_URL + 'servico.php';
        const ATENDENTES_URL = API_BASE_URL + 'servico_especialistas.php';
        const CHAMADOS_URL = API_BASE_URL + 'chamados.php';
        const CONTRATOS_URL = API_BASE_URL + 'contrato.php';
        const CONTRATO_MAQUINAS_URL = API_BASE_URL + 'contrato_maquina.php';

        // Modifique a inicialização dos eventos para isso:
        document.addEventListener('DOMContentLoaded', function() {
            carregarClientes();
            carregarTiposServico();
            
            // Configura eventos - FORMA CORRIGIDA
            const selectCliente = document.getElementById('fk_cliente');
            const selectContrato = document.getElementById('fk_contrato');
            
            selectCliente.addEventListener('change', function() {
                atualizarSolicitante();
                carregarContratos().then(() => {
                    // Força o disparo do change se já houver valor selecionado
                    if (selectContrato.value) {
                        selectContrato.dispatchEvent(new Event('change'));
                    }
                });
            });
            
            // Adiciona um observer para detectar quando o select de contrato é liberado
            const observer = new MutationObserver(function(mutations) {
                mutations.forEach(function(mutation) {
                    if (mutation.attributeName === 'disabled' && !selectContrato.disabled) {
                        selectContrato.addEventListener('change', carregarMaquinasContrato);
                    }
                });
            });
            
            observer.observe(selectContrato, { attributes: true });
            
            // Mantém os outros eventos
            document.getElementById('fk_servico_tipo').addEventListener('change', carregarServicos);
            document.getElementById('fk_servico').addEventListener('change', carregarAtendentes);
            document.getElementById('chamadoForm').addEventListener('submit', enviarChamado);
        });

        // Função para carregar clientes
        async function carregarClientes() {
            try {
                const response = await fetch(CLIENTES_URL);
                if (!response.ok) throw new Error('Erro ao carregar clientes');
                
                const clientes = await response.json();
                const select = document.getElementById('fk_cliente');
                
                // Limpa opções existentes, mantendo apenas a primeira
                select.innerHTML = '<option value="">Selecione um cliente</option>';
                
                clientes.forEach(cliente => {
                    const option = document.createElement('option');
                    option.value = cliente.id; // Persiste o ID
                    option.textContent = cliente.nome_rs; // Mostra o nome
                    select.appendChild(option);
                });
                
                // Habilita temporariamente para permitir seleção (se necessário)
                select.disabled = false;
                
            } catch (error) {
                mostrarFeedback('Erro ao carregar clientes: ' + error.message, 'danger');
            }
        }

        // Função para carregar contratos do cliente selecionado (CORRIGIDA)
        async function carregarContratos() {
            const clienteId = document.getElementById('fk_cliente').value;
            const selectContrato = document.getElementById('fk_contrato');
            
            if (!clienteId) {
                selectContrato.disabled = true;
                selectContrato.innerHTML = '<option value="">Selecione um cliente primeiro</option>';
                return;
            }
            
            try {
                // Busca TODOS os contratos (a filtragem será feita no front)
                const response = await fetch(CONTRATOS_URL);
                if (!response.ok) throw new Error('Erro ao carregar contratos');
                
                const todosContratos = await response.json();
                
                // Filtra apenas os contratos do cliente selecionado
                const contratosDoCliente = todosContratos.filter(contrato => {
                    return contrato.fk_cliente == clienteId; // Comparação flexível (==) pois pode ser string/number
                });
                
                selectContrato.innerHTML = '';
                
                if (contratosDoCliente.length === 0) {
                    selectContrato.innerHTML = '<option value="">Nenhum contrato disponível para este cliente</option>';
                } else {
                    contratosDoCliente.forEach(contrato => {
                        const option = document.createElement('option');
                        option.value = contrato.id;
                        // Mostra o ID do contrato e status (se existir)
                        option.textContent = `Contrato ${contrato.id}` + (contrato.status ? ` (${contrato.status})` : '');
                        selectContrato.appendChild(option);
                    });
                    selectContrato.disabled = false;
                }
            } catch (error) {
                mostrarFeedback('Erro ao carregar contratos: ' + error.message, 'danger');
            }
        }

        async function carregarMaquinasContrato() {
            const contratoId = document.getElementById('fk_contrato').value;
            
            if (!contratoId) {
                return;
            }

            try {
                console.log(`Buscando máquinas para contrato: ${contratoId}`); // Debug
                document.getElementById('loading').style.display = 'block';
                
                const response = await fetch(`${CONTRATO_MAQUINAS_URL}?fk_contrato=${contratoId}`);
                console.log('Resposta da API:', response); // Debug
                
                if (!response.ok) throw new Error(`HTTP error! status: ${response.status}`);
                
                const maquinas = await response.json();
                console.log('Dados das máquinas:', maquinas); // Debug
                
                if (maquinas && maquinas.length > 0) {
                    const maquina = maquinas[0];
                    console.log('Preenchendo campos com:', maquina); // Debug
                    
                    // Preenche os campos
                    document.getElementById('maquina').value = maquina.equipamento || '';
                    document.getElementById('serie').value = maquina.serie || '';
                    document.getElementById('modelo').value = maquina.modelo || '';
                    document.getElementById('setor').value = maquina.setor_area || '';
                    
                    // Endereço formatado
                    const enderecoCompleto = [
                        maquina.endereco, 
                        maquina.numero, 
                        maquina.bairro
                    ].filter(Boolean).join(' - ');
                    
                    document.getElementById('endereco').value = enderecoCompleto;
                    document.getElementById('cidade').value = maquina.cidade || '';
                    document.getElementById('uf').value = maquina.uf || '';
                    
                    // Habilita os próximos campos
                    document.getElementById('fk_servico_tipo').disabled = false;
                } else {
                    console.warn('Nenhuma máquina encontrada para este contrato');
                    mostrarFeedback('Nenhuma máquina encontrada para este contrato', 'warning');
                }
            } catch (error) {
                console.error('Erro ao carregar máquinas:', error);
                mostrarFeedback(`Erro ao carregar máquinas: ${error.message}`, 'danger');
            } finally {
                document.getElementById('loading').style.display = 'none';
            }
        }

        // Função para carregar tipos de serviço
        async function carregarTiposServico() {
            try {
                const response = await fetch(SERVICOS_TIPO_URL);
                if (!response.ok) throw new Error('Erro ao carregar tipos de serviço');
                
                const tipos = await response.json();
                const select = document.getElementById('fk_servico_tipo');
                
                tipos.forEach(tipo => {
                    const option = document.createElement('option');
                    option.value = tipo.id;
                    option.textContent = tipo.descricao;
                    select.appendChild(option);
                });
            } catch (error) {
                mostrarFeedback('Erro ao carregar tipos de serviço: ' + error.message, 'danger');
            }
        }

        // Função para carregar serviços baseados no tipo selecionado
        async function carregarServicos() {
            const tipoId = document.getElementById('fk_servico_tipo').value;
            const selectServico = document.getElementById('fk_servico');
            
            if (!tipoId) {
                selectServico.disabled = true;
                selectServico.innerHTML = '<option value="">Selecione o tipo primeiro</option>';
                return;
            }
            
            try {
                const response = await fetch(`${SERVICOS_URL}?tipo_id=${tipoId}`);
                if (!response.ok) throw new Error('Erro ao carregar serviços');
                
                const servicos = await response.json();
                selectServico.innerHTML = '';
                
                if (servicos.length === 0) {
                    selectServico.innerHTML = '<option value="">Nenhum serviço disponível</option>';
                } else {
                    servicos.forEach(servico => {
                        const option = document.createElement('option');
                        option.value = servico.id;
                        option.textContent = servico.descricao;
                        selectServico.appendChild(option);
                    });
                    selectServico.disabled = false;
                }
                
                // Limpa atendentes quando muda o serviço
                document.getElementById('fk_atendente').innerHTML = '<option value="">Selecione o serviço primeiro</option>';
                document.getElementById('fk_atendente').disabled = true;
            } catch (error) {
                mostrarFeedback('Erro ao carregar serviços: ' + error.message, 'danger');
            }
        }

        // Função para carregar atendentes agrupando apenas pela descrição
        async function carregarAtendentes() {
            const servicoId = document.getElementById('fk_servico').value;
            const selectAtendente = document.getElementById('fk_atendente');
            
            if (!servicoId) {
                selectAtendente.disabled = true;
                selectAtendente.innerHTML = '<option value="">Selecione o serviço primeiro</option>';
                return;
            }
            
            try {
                const response = await fetch(`${ATENDENTES_URL}?servico_id=${servicoId}`);
                if (!response.ok) throw new Error('Erro ao carregar atendentes');
                
                const atendentes = await response.json();
                selectAtendente.innerHTML = '';
                
                if (atendentes.length === 0) {
                    selectAtendente.innerHTML = '<option value="">Nenhum atendente disponível</option>';
                } else {
                    // Objeto para armazenar descrições únicas
                    const descricoesUnicas = {};
                    
                    // Filtra atendentes mantendo apenas a primeira ocorrência de cada descrição
                    const atendentesUnicos = atendentes.filter(atendente => {
                        if (!descricoesUnicas[atendente.descricao]) {
                            descricoesUnicas[atendente.descricao] = true;
                            return true;
                        }
                        return false;
                    });
                    
                    // Ordena por descrição (nome do atendente)
                    atendentesUnicos.sort((a, b) => a.descricao.localeCompare(b.descricao));
                    
                    // Adiciona as opções no select
                    atendentesUnicos.forEach(atendente => {
                        const option = document.createElement('option');
                        // Mantemos o value como id mesmo agrupando por descrição
                        option.value = atendente.id; 
                        option.textContent = atendente.descricao;
                        selectAtendente.appendChild(option);
                    });
                    
                    selectAtendente.disabled = false;
                }
            } catch (error) {
                mostrarFeedback('Erro ao carregar atendentes: ' + error.message, 'danger');
            }
        }

        // Atualiza o campo solicitante quando seleciona um cliente
        function atualizarSolicitante() {
            const selectCliente = document.getElementById('fk_cliente');
            const solicitante = document.getElementById('cliente_solicitante');
            
            if (selectCliente.selectedIndex > 0) {
                solicitante.value = selectCliente.options[selectCliente.selectedIndex].text;
            } else {
                solicitante.value = '';
            }
        }

        // Envia o formulário para a API
        async function enviarChamado(event) {
            event.preventDefault();
            
            // Validação básica
            if (!document.getElementById('chamadoForm').checkValidity()) {
                mostrarFeedback('Preencha todos os campos obrigatórios', 'warning');
                return;
            }
            
            // Verifica se o cliente está selecionado
            const clienteId = document.getElementById('fk_cliente').value;
            if (!clienteId) {
                mostrarFeedback('Selecione um cliente válido', 'warning');
                return;
            }
            
            // Mostra loading
            document.getElementById('loading').style.display = 'block';
            document.getElementById('feedback').innerHTML = '';
            
            try {
                // Prepara os dados
                const formData = {
                    fk_cliente: clienteId,
                    cliente_solicitante: document.getElementById('cliente_solicitante').value,
                    fk_contrato: document.getElementById('fk_contrato').value,
                    serie: document.getElementById('serie').value,
                    modelo: document.getElementById('modelo').value,
                    maquina: document.getElementById('maquina').value,
                    setor: document.getElementById('setor').value,
                    endereco: document.getElementById('endereco').value,
                    cidade: document.getElementById('cidade').value,
                    uf: document.getElementById('uf').value,
                    fk_servico_tipo: document.getElementById('fk_servico_tipo').value,
                    fk_servico: document.getElementById('fk_servico').value,
                    fk_atendente: document.getElementById('fk_atendente').value,
                    detalhes: document.getElementById('detalhes').value,
                    anexo: null
                };
                
                console.log('Dados sendo enviados:', formData); // Debug
                
                // Envia para a API
                const response = await fetch(CHAMADOS_URL, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(formData)
                });
                
                if (!response.ok) {
                    const errorData = await response.json().catch(() => ({}));
                    throw new Error(errorData.error || `Erro HTTP: ${response.status}`);
                }
                
                const data = await response.json();
                mostrarFeedback('Chamado aberto com sucesso! Número: ' + data.id, 'success');
                limparFormulario();
            } catch (error) {
                console.error('Erro detalhado:', error);
                mostrarFeedback('Erro ao abrir chamado: ' + error.message, 'danger');
            } finally {
                document.getElementById('loading').style.display = 'none';
            }
        }

        // Limpa o formulário
        function limparFormulario() {
            document.getElementById('chamadoForm').reset();
            document.getElementById('fk_contrato').disabled = true;
            document.getElementById('fk_contrato').innerHTML = '<option value="">Selecione um cliente primeiro</option>';
            document.getElementById('fk_servico').disabled = true;
            document.getElementById('fk_servico').innerHTML = '<option value="">Selecione o tipo primeiro</option>';
            document.getElementById('fk_atendente').disabled = true;
            document.getElementById('fk_atendente').innerHTML = '<option value="">Selecione o serviço primeiro</option>';
            document.getElementById('feedback').innerHTML = '';
            const selectCliente = document.getElementById('fk_cliente');
            selectCliente.selectedIndex = 0;
            selectCliente.disabled = true;
        }

        // Mostra mensagem de feedback
        function mostrarFeedback(mensagem, tipo) {
            const feedbackDiv = document.getElementById('feedback');
            feedbackDiv.innerHTML = `
                <div class="alert alert-${tipo} alert-dismissible fade show" role="alert">
                    ${mensagem}
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            `;
        }





        // Adicione estas constantes no início das variáveis
        const ITENS_POR_PAGINA = 10;
        let paginaAtual = 1;
        let totalRegistros = 0;
        let todasMaquinas = [];

        // Adicione este código no DOMContentLoaded
        document.getElementById('btnAbrirBuscaMaquina').addEventListener('click', abrirModalBusca);
        document.getElementById('btnBuscarMaquinas').addEventListener('click', buscarMaquinas);
        document.getElementById('inputBuscaMaquina').addEventListener('keyup', function(e) {
            if (e.key === 'Enter') buscarMaquinas();
        });

        // Funções para o modal de busca
        function abrirModalBusca() {
            // Limpa a busca anterior
            document.getElementById('inputBuscaMaquina').value = '';
            paginaAtual = 1;
            
            // Abre o modal
            const modal = new bootstrap.Modal(document.getElementById('modalBuscaMaquina'));
            modal.show();
            
            // Carrega os dados
            buscarMaquinas();
        }

        async function buscarMaquinas() {
            const termoBusca = document.getElementById('inputBuscaMaquina').value.toLowerCase();
            
            try {
                document.getElementById('loading').style.display = 'block';
                const response = await fetch(CONTRATO_MAQUINAS_URL);
                
                if (!response.ok) throw new Error('Erro ao buscar máquinas');
                
                todasMaquinas = await response.json();
                totalRegistros = todasMaquinas.length;
                
                // Filtra os resultados se houver termo de busca
                let maquinasFiltradas = todasMaquinas;
                if (termoBusca) {
                    maquinasFiltradas = todasMaquinas.filter(maquina => {
                        return (
                            (maquina.equipamento && maquina.equipamento.toLowerCase().includes(termoBusca)) ||
                            (maquina.modelo && maquina.modelo.toLowerCase().includes(termoBusca)) ||
                            (maquina.serie && maquina.serie.toLowerCase().includes(termoBusca)) ||
                            (maquina.fk_nome_matriz && maquina.fk_nome_matriz.toLowerCase().includes(termoBusca))
                        );
                    });
                }
                
                totalRegistros = maquinasFiltradas.length;
                atualizarTabelaMaquinas(maquinasFiltradas);
                atualizarPaginacao(maquinasFiltradas.length);
                
            } catch (error) {
                console.error('Erro ao buscar máquinas:', error);
                mostrarFeedback('Erro ao buscar máquinas: ' + error.message, 'danger');
            } finally {
                document.getElementById('loading').style.display = 'none';
            }
        }

        function atualizarTabelaMaquinas(maquinas) {
            const inicio = (paginaAtual - 1) * ITENS_POR_PAGINA;
            const fim = inicio + ITENS_POR_PAGINA;
            const maquinasPagina = maquinas.slice(inicio, fim);
            
            const corpoTabela = document.getElementById('corpoTabelaMaquinas');
            corpoTabela.innerHTML = '';
            
            if (maquinasPagina.length === 0) {
                corpoTabela.innerHTML = '<tr><td colspan="9" class="text-center">Nenhuma máquina encontrada</td></tr>';
                return;
            }
            
            maquinasPagina.forEach(maquina => {
                const linha = document.createElement('tr');
                linha.innerHTML = `
                    <td>${maquina.fk_contrato || ''}</td>
                    <td>${maquina.fk_nome_matriz || ''}</td>
                    <td>${maquina.equipamento || ''}</td>
                    <td>${maquina.modelo || ''}</td>
                    <td>${maquina.serie || ''}</td>
                    <td>${maquina.setor_area || ''}</td>
                    <td>${maquina.cidade || ''}</td>
                    <td>${maquina.uf || ''}</td>
                    <td><button class="btn btn-sm btn-primary btn-selecionar" data-maquina='${JSON.stringify(maquina)}'>Selecionar</button></td>
                `;
                corpoTabela.appendChild(linha);
            });
            
            // Adiciona eventos aos botões de seleção
            document.querySelectorAll('.btn-selecionar').forEach(btn => {
                btn.addEventListener('click', function() {
                    const maquina = JSON.parse(this.getAttribute('data-maquina'));
                    preencherCamposComMaquina(maquina);
                    bootstrap.Modal.getInstance(document.getElementById('modalBuscaMaquina')).hide();
                });
            });
            
            // Atualiza informações de paginação
            document.getElementById('infoPaginacao').textContent = 
                `Mostrando ${inicio + 1} a ${Math.min(fim, totalRegistros)} de ${totalRegistros} registros`;
        }

        function atualizarPaginacao(totalItens) {
            const totalPaginas = Math.ceil(totalItens / ITENS_POR_PAGINA);
            const paginacao = document.getElementById('paginacaoMaquinas');
            paginacao.innerHTML = '';
            
            if (totalPaginas <= 1) return;

            // Configurações de paginação
            const maxPaginasVisiveis = 5; // Número máximo de páginas visíveis
            let inicio = 1;
            let fim = totalPaginas;

            // Calcula o intervalo de páginas a serem mostradas
            if (totalPaginas > maxPaginasVisiveis) {
                inicio = Math.max(1, paginaAtual - Math.floor(maxPaginasVisiveis / 2));
                fim = inicio + maxPaginasVisiveis - 1;
                
                if (fim > totalPaginas) {
                    fim = totalPaginas;
                    inicio = fim - maxPaginasVisiveis + 1;
                }
            }

            // Botão Anterior
            const liAnterior = document.createElement('li');
            liAnterior.className = `page-item ${paginaAtual === 1 ? 'disabled' : ''}`;
            liAnterior.innerHTML = `<a class="page-link" href="#" aria-label="Anterior">
                <span aria-hidden="true">&laquo;</span>
            </a>`;
            liAnterior.addEventListener('click', (e) => {
                e.preventDefault();
                if (paginaAtual > 1) {
                    paginaAtual--;
                    buscarMaquinas();
                }
            });
            paginacao.appendChild(liAnterior);

            // Primeira página + "..." se necessário
            if (inicio > 1) {
                const liPrimeira = document.createElement('li');
                liPrimeira.className = 'page-item';
                liPrimeira.innerHTML = `<a class="page-link" href="#">1</a>`;
                liPrimeira.addEventListener('click', (e) => {
                    e.preventDefault();
                    paginaAtual = 1;
                    buscarMaquinas();
                });
                paginacao.appendChild(liPrimeira);

                if (inicio > 2) {
                    const liEllipsis = document.createElement('li');
                    liEllipsis.className = 'page-item disabled';
                    liEllipsis.innerHTML = `<span class="page-link">...</span>`;
                    paginacao.appendChild(liEllipsis);
                }
            }

            // Páginas numeradas
            for (let i = inicio; i <= fim; i++) {
                const li = document.createElement('li');
                li.className = `page-item ${i === paginaAtual ? 'active' : ''}`;
                li.innerHTML = `<a class="page-link" href="#">${i}</a>`;
                li.addEventListener('click', (e) => {
                    e.preventDefault();
                    paginaAtual = i;
                    buscarMaquinas();
                });
                paginacao.appendChild(li);
            }

            // Última página + "..." se necessário
            if (fim < totalPaginas) {
                if (fim < totalPaginas - 1) {
                    const liEllipsis = document.createElement('li');
                    liEllipsis.className = 'page-item disabled';
                    liEllipsis.innerHTML = `<span class="page-link">...</span>`;
                    paginacao.appendChild(liEllipsis);
                }

                const liUltima = document.createElement('li');
                liUltima.className = 'page-item';
                liUltima.innerHTML = `<a class="page-link" href="#">${totalPaginas}</a>`;
                liUltima.addEventListener('click', (e) => {
                    e.preventDefault();
                    paginaAtual = totalPaginas;
                    buscarMaquinas();
                });
                paginacao.appendChild(liUltima);
            }

            // Botão Próximo
            const liProximo = document.createElement('li');
            liProximo.className = `page-item ${paginaAtual === totalPaginas ? 'disabled' : ''}`;
            liProximo.innerHTML = `<a class="page-link" href="#" aria-label="Próximo">
                <span aria-hidden="true">&raquo;</span>
            </a>`;
            liProximo.addEventListener('click', (e) => {
                e.preventDefault();
                if (paginaAtual < totalPaginas) {
                    paginaAtual++;
                    buscarMaquinas();
                }
            });
            paginacao.appendChild(liProximo);

            // Atualiza informações de paginação
            document.getElementById('infoPaginacao').textContent = 
                `Mostrando ${Math.min((paginaAtual - 1) * ITENS_POR_PAGINA + 1, totalItens)} a ${Math.min(paginaAtual * ITENS_POR_PAGINA, totalItens)} de ${totalItens} registros`;
        }

        function habilitarSelecaoCliente() {
            const select = document.getElementById('fk_cliente');
            select.disabled = false;
            select.focus(); // Opcional: dá foco ao campo
        }

        // Exemplo de uso (pode ser chamado por um botão "Editar" ou similar)
        document.getElementById('btnEditarCliente').addEventListener('click', habilitarSelecaoCliente);


        async function preencherCamposComMaquina(maquina) {
            try {
                // 1. Preenche os campos básicos da máquina
                document.getElementById('maquina').value = maquina.equipamento || '';
                document.getElementById('serie').value = maquina.serie || '';
                document.getElementById('modelo').value = maquina.modelo || '';
                document.getElementById('setor').value = maquina.setor_area || '';
                
                // Formata o endereço
                const enderecoParts = [
                    maquina.endereco,
                    maquina.numero,
                    maquina.bairro
                ].filter(part => part !== null && part !== undefined && part !== '');
                
                document.getElementById('endereco').value = enderecoParts.join(' - ');
                document.getElementById('cidade').value = maquina.cidade || '';
                document.getElementById('uf').value = maquina.uf || '';

                // 2. Se não tiver fk_contrato, limpa os campos e retorna
                if (!maquina.fk_contrato) {
                    document.getElementById('fk_contrato').value = '';
                    document.getElementById('fk_cliente').value = '';
                    document.getElementById('fk_cliente').selectedIndex = 0;
                    return;
                }

                // 3. Preenche o contrato
                document.getElementById('fk_contrato').value = maquina.fk_contrato;
                
                // 4. Busca os dados do contrato
                let contrato;
                try {
                    const response = await fetch(`${CONTRATOS_URL}?id=${maquina.fk_contrato}`);
                    if (!response.ok) throw new Error('Erro na resposta da API');
                    
                    const data = await response.json();
                    
                    // Verifica o formato da resposta
                    if (Array.isArray(data)) {
                        if (data.length === 0) throw new Error('Contrato não encontrado');
                        contrato = data[0];
                    } else if (typeof data === 'object') {
                        contrato = data;
                    } else {
                        throw new Error('Formato de resposta inesperado');
                    }
                    
                    if (!contrato.hasOwnProperty('fk_cliente')) {
                        throw new Error('Contrato não possui fk_cliente');
                    }
                    
                } catch (error) {
                    console.error('Erro ao buscar contrato:', error);
                    document.getElementById('fk_cliente').selectedIndex = 0;
                    throw new Error('Não foi possível carregar os dados do contrato');
                }

                // 5. Busca e preenche os dados do cliente
                try {
                    const response = await fetch(`${CLIENTES_URL}?id=${contrato.fk_cliente}`);
                    if (!response.ok) throw new Error('Erro na resposta da API');
                    
                    const data = await response.json();
                    let cliente;
                    
                    // Verifica o formato da resposta
                    if (Array.isArray(data)) {
                        if (data.length === 0) throw new Error('Cliente não encontrado');
                        cliente = data[0];
                    } else if (typeof data === 'object') {
                        cliente = data;
                    } else {
                        throw new Error('Formato de resposta inesperado');
                    }
                    
                    if (!cliente.hasOwnProperty('id') || !cliente.hasOwnProperty('nome_rs')) {
                        throw new Error('Dados incompletos do cliente');
                    }
                    
                    // Preenche o select de cliente
                    const selectCliente = document.getElementById('fk_cliente');
                    
                    // Verifica se o cliente já está nas opções
                    let clienteExiste = false;
                    for (let i = 0; i < selectCliente.options.length; i++) {
                        if (selectCliente.options[i].value == cliente.id) {
                            selectCliente.selectedIndex = i;
                            clienteExiste = true;
                            break;
                        }
                    }
                    
                    // Se não encontrou, adiciona uma nova opção
                    if (!clienteExiste) {
                        const option = document.createElement('option');
                        option.value = cliente.id;
                        option.textContent = cliente.nome_rs;
                        selectCliente.appendChild(option);
                        selectCliente.selectedIndex = selectCliente.options.length - 1;
                    }
                    
                    // Atualiza o solicitante automaticamente
                    document.getElementById('cliente_solicitante').value = cliente.nome_rs;
                    
                } catch (error) {
                    console.error('Erro ao buscar cliente:', error);
                    document.getElementById('fk_cliente').selectedIndex = 0;
                    throw new Error('Não foi possível carregar os dados do cliente');
                }
                
                // Habilita os próximos campos
                document.getElementById('fk_servico_tipo').disabled = false;
                
                // Fecha o modal
                const modal = bootstrap.Modal.getInstance(document.getElementById('modalBuscaMaquina'));
                if (modal) modal.hide();
                
            } catch (error) {
                console.error('Erro no preenchimento:', error);
                mostrarFeedback(error.message, 'warning');
            }
        }

        async function buscarClienteEContratoPorMaquina(maquina) {
            try {
                if (!maquina.fk_contrato) return;
                
                // Busca o contrato
                const responseContrato = await fetch(`${CONTRATOS_URL}?id=${maquina.fk_contrato}`);
                if (!responseContrato.ok) throw new Error('Erro ao buscar contrato');
                
                const contratos = await responseContrato.json();
                if (contratos.length === 0) return;
                
                const contrato = contratos[0];
                
                // Busca o cliente
                const responseCliente = await fetch(`${CLIENTES_URL}?id=${contrato.fk_cliente}`);
                if (!responseCliente.ok) throw new Error('Erro ao buscar cliente');
                
                const clientes = await responseCliente.json();
                if (clientes.length === 0) return;
                
                const cliente = clientes[0];
                
                // Preenche os campos de cliente e contrato
                const selectCliente = document.getElementById('fk_cliente');
                const selectContrato = document.getElementById('fk_contrato');
                
                // Seleciona o cliente
                for (let i = 0; i < selectCliente.options.length; i++) {
                    if (selectCliente.options[i].value == cliente.id) {
                        selectCliente.selectedIndex = i;
                        selectCliente.dispatchEvent(new Event('change'));
                        break;
                    }
                }
                
                // Aguarda o carregamento dos contratos
                setTimeout(() => {
                    // Seleciona o contrato
                    for (let i = 0; i < selectContrato.options.length; i++) {
                        if (selectContrato.options[i].value == contrato.id) {
                            selectContrato.selectedIndex = i;
                            selectContrato.dispatchEvent(new Event('change'));
                            break;
                        }
                    }
                }, 500);
                
            } catch (error) {
                console.error('Erro ao buscar cliente/contrato:', error);
            }
        }



    </script>
</body>
</html>