import os
import json

def generate_crud():
    print("Script de geração de CRUD Frontend")
    print("----------------------------------\n")
    
    # Perguntar onde salvar o script
    save_path = input("Onde deseja salvar o script? (ex: modules/cadastro/): ").strip()
    if not save_path.endswith('/'):
        save_path += '/'
    
    # Criar diretório se não existir
    os.makedirs(save_path, exist_ok=True)
    
    # Perguntar pelo endpoint
    endpoint = input("\nQual o endpoint? (ex: http://localhost/jps-service/backend/routers/produto_ncm.php): ").strip()
    
    # Perguntar pelo JSON de resultado
    print("\nQual o json de resultado? (digite END em uma nova linha para finalizar):")
    json_lines = []
    while True:
        line = input()
        if line.strip().upper() == 'END':
            break
        json_lines.append(line)
    
    json_str = '\n'.join(json_lines)
    try:
        example_data = json.loads(json_str)
        if isinstance(example_data, list):
            example_data = example_data[0] if len(example_data) > 0 else {}
    except json.JSONDecodeError:
        print("JSON inválido. Usando objeto vazio.")
        example_data = {}
    
    # Extrair nome da entidade do endpoint (para título da página)
    entity_name = endpoint.split('/')[-1].replace('.php', '').replace('_', ' ').title()
    
    # Gerar nome do arquivo
    filename = f"{save_path}{entity_name.replace(' ', '_').lower()}.php"
    
    # Gerar colunas da tabela com base no JSON, removendo 'id' dos formulários
    columns = []
    form_columns = []  # Colunas que aparecerão nos formulários (excluindo 'id')
    for key in example_data.keys():
        columns.append(key)
        if key.lower() != 'id':  # Não incluir 'id' nos formulários
            form_columns.append(key)
    
    # Gerar o conteúdo do arquivo PHP
    php_content = f"""<?php
$api_url = '{endpoint}';
$data = json_decode(file_get_contents($api_url), true);

// Função para formatar valores se necessário
function formatValue($value) {{
    if (is_numeric($value)) {{
        return number_format(floatval($value), 2, ',', '.');
    }}
    return $value;
}}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestão de {entity_name}</title>
    <link rel="stylesheet" type="text/css" href="https://cdn.datatables.net/v/bs5/jq-3.6.0/dt-1.11.5/datatables.min.css"/>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .btn-action {{
            padding: 0.25rem 0.5rem;
            font-size: 0.875rem;
            line-height: 1.5;
        }}
        .modal-lg-custom {{
            max-width: 800px;
        }}
    </style>
</head>
<body>
    <div class="container-fluid mt-4">
        <h1 class="mb-4">Gestão de {entity_name}</h1>
       
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">Lista de {entity_name}</h5>
                <div>
                    <button class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#cadastrarModal">
                        <i class="fas fa-plus me-1"></i> Novo Registro
                    </button>
                </div>
            </div>
            <div class="card-body">
                <table id="dataTable" class="table table-striped table-bordered nowrap" style="width:100%">
                    <thead>
                        <tr>
                            {"".join(f'<th>{col.replace("_", " ").title()}</th>' for col in columns)}
                            <th>Ações</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($data as $item): ?>
                        <tr>
                            {"".join(f'<td><?= formatValue($item["{col}"]) ?></td>' for col in columns)}
                            <td>
                                <button class="btn btn-info btn-action btn-view" data-id="<?= $item["id"] ?>">
                                    <i class="fas fa-eye"></i>
                                </button>
                                <button class="btn btn-warning btn-action btn-edit" data-id="<?= $item["id"] ?>">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button class="btn btn-danger btn-action btn-delete" data-id="<?= $item["id"] ?>">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- Modal Visualizar -->
    <div class="modal fade" id="viewModal" tabindex="-1" aria-labelledby="viewModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg modal-lg-custom">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="viewModalLabel">Detalhes</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body" id="viewModalBody">
                    <!-- Conteúdo será preenchido via AJAX -->
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Fechar</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal Editar -->
    <div class="modal fade" id="editModal" tabindex="-1" aria-labelledby="editModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg modal-lg-custom">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="editModalLabel">Editar</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form id="editForm" method="POST">
                    <div class="modal-body" id="editModalBody">
                        <!-- Conteúdo será preenchido via AJAX -->
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" class="btn btn-primary">Salvar Alterações</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Modal Excluir -->
    <div class="modal fade" id="deleteModal" tabindex="-1" aria-labelledby="deleteModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="deleteModalLabel">Confirmar Exclusão</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <p>Você tem certeza que deseja excluir este registro? Esta ação não pode ser desfeita.</p>
                    <input type="hidden" id="deleteId">
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="button" class="btn btn-danger" id="confirmDelete">Excluir</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal Cadastrar -->
    <div class="modal fade" id="cadastrarModal" tabindex="-1" aria-labelledby="cadastrarModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg modal-lg-custom">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="cadastrarModalLabel">Cadastrar Novo</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form id="cadastrarForm" method="POST">
                    <div class="modal-body">
                        <div class="row g-3">
                            {"".join(f'''
                            <div class="col-md-6">
                                <label for="{col}" class="form-label">{col.replace("_", " ").title()}</label>
                                <input type="text" class="form-control" id="{col}" name="{col}" required>
                            </div>
                            ''' for col in form_columns)}
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" class="btn btn-primary">Cadastrar</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- jQuery, Bootstrap e DataTables JS -->
    <script type="text/javascript" src="https://cdn.datatables.net/v/bs5/jq-3.6.0/dt-1.11.5/datatables.min.js"></script>
    <!-- Excel export -->
    <script src="https://cdn.datatables.net/buttons/2.2.2/js/dataTables.buttons.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jszip/3.1.3/jszip.min.js"></script>
    <script src="https://cdn.datatables.net/buttons/2.2.2/js/buttons.html5.min.js"></script>

    <script>
        $(document).ready(function() {{
            // Inicialização do DataTable com botão de exportação
            var table = $('#dataTable').DataTable({{
                responsive: true,
                dom: 'Bfrtip',
                buttons: [
                    {{
                        extend: 'excel',
                        text: '<i class="fas fa-file-excel me-1"></i> Exportar Excel',
                        className: 'btn btn-success btn-sm',
                        title: '{entity_name}',
                        exportOptions: {{
                            columns: ':not(:last-child)' // Exclui a coluna de ações
                        }}
                    }}
                ],
                language: {{
                    url: 'https://cdn.datatables.net/plug-ins/1.11.5/i18n/pt-BR.json'
                }}
            }});

            // Visualizar registro
            $(document).on('click', '.btn-view', function() {{
                var id = $(this).data('id');
                $.ajax({{
                    url: '<?= $api_url ?>?id=' + id,
                    type: 'GET',
                    dataType: 'json',
                    success: function(item) {{
                        var html = '<div class="row">';
                        {''.join(f'''html += `
                            <div class="col-md-6">
                                <p><strong>{col.replace('_', ' ').title()}:</strong> ${{item['{col}'] || 'N/A'}}</p>
                            </div>
                        `;''' for col in columns)}
                        html += '</div>';
                        $('#viewModalBody').html(html);
                        $('#viewModal').modal('show');
                    }},
                    error: function(xhr, status, error) {{
                        alert('Erro ao carregar registro: ' + error);
                    }}
                }});
            }});

            // Editar registro - Abrir modal
            $(document).on('click', '.btn-edit', function() {{
                var id = $(this).data('id');
                $.ajax({{
                    url: '<?= $api_url ?>?id=' + id,
                    type: 'GET',
                    dataType: 'json',
                    success: function(item) {{
                        var html = '<input type="hidden" name="id" value="' + item.id + '">';
                        html += '<div class="row g-3">';
                        {''.join(f'''html += `
                            <div class="col-md-6">
                                <label for="edit_{col}" class="form-label">{col.replace('_', ' ').title()}</label>
                                <input type="text" class="form-control" id="edit_{col}" name="{col}" value="${{item['{col}'] || ''}}" required>
                            </div>
                        `;''' for col in form_columns)}
                        html += '</div>';
                        $('#editModalBody').html(html);
                        $('#editModal').modal('show');
                    }},
                    error: function(xhr, status, error) {{
                        alert('Erro ao carregar registro para edição: ' + error);
                    }}
                }});
            }});

            // Submit do formulário de edição
            $('#editForm').submit(function(e) {{
                e.preventDefault();
                var id = $('input[name="id"]').val();
                var formData = {{
                    {', '.join(f'"{col}": $("#edit_{col}").val()' for col in form_columns)}
                }};
               
                $.ajax({{
                    url: '<?= $api_url ?>?id=' + id,
                    type: 'PUT',
                    contentType: 'application/json',
                    data: JSON.stringify(formData),
                    success: function(response) {{
                        alert('Registro atualizado com sucesso!');
                        $('#editModal').modal('hide');
                        location.reload();
                    }},
                    error: function(xhr, status, error) {{
                        alert('Erro ao atualizar o registro: ' + (xhr.responseJSON?.error || error));
                    }}
                }});
            }});

            // Excluir registro
            $(document).on('click', '.btn-delete', function() {{
                var id = $(this).data('id');
                $('#deleteId').val(id);
                $('#deleteModal').modal('show');
            }});

            // Confirmar exclusão
            $('#confirmDelete').click(function() {{
                var id = $('#deleteId').val();
               
                $.ajax({{
                    url: '<?= $api_url ?>?id=' + id,
                    type: 'DELETE',
                    success: function(response) {{
                        alert('Registro excluído com sucesso!');
                        $('#deleteModal').modal('hide');
                        location.reload();
                    }},
                    error: function(xhr, status, error) {{
                        alert('Erro ao excluir o registro: ' + (xhr.responseJSON?.error || error));
                    }}
                }});
            }});

            // Submit do formulário de cadastro
            $('#cadastrarForm').submit(function(e) {{
                e.preventDefault();
                var formData = {{
                    {', '.join(f'"{col}": $("#{col}").val()' for col in form_columns)}
                }};
               
                $.ajax({{
                    url: '<?= $api_url ?>',
                    type: 'POST',
                    contentType: 'application/json',
                    data: JSON.stringify(formData),
                    success: function(response) {{
                        alert('Registro cadastrado com sucesso!');
                        $('#cadastrarModal').modal('hide');
                        location.reload();
                    }},
                    error: function(xhr, status, error) {{
                        alert('Erro ao cadastrar o registro: ' + (xhr.responseJSON?.error || error));
                    }}
                }});
            }});
        }});
    </script>
</body>
</html>
"""
    
    # Salvar o arquivo
    with open(filename, 'w', encoding='utf-8') as file:
        file.write(php_content)
    
    print(f"\nArquivo gerado com sucesso: {filename}")

if __name__ == "__main__":
    generate_crud()