<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type");

// Conexao com o .env
$dotenv = parse_ini_file(__DIR__ . '../../.env');
$dbHost = $dotenv['DB_HOST'];
$dbUser = $dotenv['DB_USER'];
$dbPass = $dotenv['DB_PASSWORD'];
$dbName = $dotenv['DB_NAME'];

// Conexao com o banco de dados
try {
    $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha na conexao com o banco de dados: ' . $e->getMessage()]);
    exit;
}

// Roteamento
$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;

// Verifica se o script esta sendo acessado diretamente
if (basename($_SERVER['SCRIPT_FILENAME']) === 'usuarios.php') {
    switch ($method) {
        case 'GET':
            if ($id) {
                getUsuario($pdo, $id);
            } else {
                getAllUsuarios($pdo);
            }
            break;
        case 'POST':
            createUsuario($pdo);
            break;
        case 'PUT':
            if ($id) {
                updateUsuario($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do usuario nao especificado']);
            }
            break;
        case 'DELETE':
            if ($id) {
                deleteUsuario($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do usuario nao especificado']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Metodo nao permitido']);
    }
} else {
    http_response_code(404);
    echo json_encode(['error' => 'Endpoint nao encontrado']);
}

// Funcoes CRUD

function getAllUsuarios($pdo) {
    try {
        $stmt = $pdo->query("SELECT * FROM usuario");
        $usuarios = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($usuarios);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar usuarios: ' . $e->getMessage()]);
    }
}

function getUsuario($pdo, $id) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM usuario WHERE id = ?");
        $stmt->execute([$id]);
        $usuario = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($usuario) {
            echo json_encode($usuario);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Usuario nao encontrado']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar usuario: ' . $e->getMessage()]);
    }
}

function createUsuario($pdo) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados invalidos']);
        return;
    }
    
    if (!isset($data['foto'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "foto" faltando']);
        return;
    }
    if (strlen($data['foto']) > 255) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "foto" deve ter no maximo 255 caracteres']);
        return;
    }

    if (!isset($data['login'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "login" faltando']);
        return;
    }
    if (strlen($data['login']) > 20) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "login" deve ter no maximo 20 caracteres']);
        return;
    }

    if (!isset($data['senha'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "senha" faltando']);
        return;
    }
    if (strlen($data['senha']) > 20) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "senha" deve ter no maximo 20 caracteres']);
        return;
    }

    if (!isset($data['nome'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "nome" faltando']);
        return;
    }
    if (strlen($data['nome']) > 50) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "nome" deve ter no maximo 50 caracteres']);
        return;
    }

    if (!isset($data['cpf'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "cpf" faltando']);
        return;
    }
    if (strlen($data['cpf']) > 20) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "cpf" deve ter no maximo 20 caracteres']);
        return;
    }

    if (!isset($data['cel'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "cel" faltando']);
        return;
    }
    if (strlen($data['cel']) > 20) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "cel" deve ter no maximo 20 caracteres']);
        return;
    }

    if (!isset($data['email'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "email" faltando']);
        return;
    }
    if (strlen($data['email']) > 255) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "email" deve ter no maximo 255 caracteres']);
        return;
    }


    try {
        $sql = "INSERT INTO usuario (`foto`, `login`, `senha`, `nome`, `cpf`, `cel`, `email`) VALUES (:foto, :login, :senha, :nome, :cpf, :cel, :email)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':foto' => $data['foto'], ':login' => $data['login'], ':senha' => $data['senha'], ':nome' => $data['nome'], ':cpf' => $data['cpf'], ':cel' => $data['cel'], ':email' => $data['email']
        ]);
        
        $id = $pdo->lastInsertId();
        http_response_code(201);
        echo json_encode(['id' => $id, 'message' => 'Usuario criado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao criar usuario: ' . $e->getMessage()]);
    }
}

function updateUsuario($pdo, $id) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados invalidos']);
        return;
    }
    
    if (!isset($data['foto'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "foto" faltando']);
        return;
    }
    if (strlen($data['foto']) > 255) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "foto" deve ter no maximo 255 caracteres']);
        return;
    }

    if (!isset($data['login'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "login" faltando']);
        return;
    }
    if (strlen($data['login']) > 20) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "login" deve ter no maximo 20 caracteres']);
        return;
    }

    if (!isset($data['senha'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "senha" faltando']);
        return;
    }
    if (strlen($data['senha']) > 20) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "senha" deve ter no maximo 20 caracteres']);
        return;
    }

    if (!isset($data['nome'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "nome" faltando']);
        return;
    }
    if (strlen($data['nome']) > 50) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "nome" deve ter no maximo 50 caracteres']);
        return;
    }

    if (!isset($data['cpf'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "cpf" faltando']);
        return;
    }
    if (strlen($data['cpf']) > 20) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "cpf" deve ter no maximo 20 caracteres']);
        return;
    }

    if (!isset($data['cel'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "cel" faltando']);
        return;
    }
    if (strlen($data['cel']) > 20) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "cel" deve ter no maximo 20 caracteres']);
        return;
    }

    if (!isset($data['email'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "email" faltando']);
        return;
    }
    if (strlen($data['email']) > 255) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "email" deve ter no maximo 255 caracteres']);
        return;
    }


    try {
        // Verifica se o usuario existe
        $check = $pdo->prepare("SELECT id FROM usuario WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Usuario nao encontrado']);
            return;
        }
        
        $sql = "UPDATE usuario SET `foto` = :foto, `login` = :login, `senha` = :senha, `nome` = :nome, `cpf` = :cpf, `cel` = :cel, `email` = :email WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':foto' => $data['foto'], ':login' => $data['login'], ':senha' => $data['senha'], ':nome' => $data['nome'], ':cpf' => $data['cpf'], ':cel' => $data['cel'], ':email' => $data['email'], ':id' => $id
        ]);
        
        echo json_encode(['message' => 'Usuario atualizado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao atualizar usuario: ' . $e->getMessage()]);
    }
}

function deleteUsuario($pdo, $id) {
    try {
        // Verifica se o usuario existe
        $check = $pdo->prepare("SELECT id FROM usuario WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Usuario nao encontrado']);
            return;
        }
        
        $stmt = $pdo->prepare("DELETE FROM usuario WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode(['message' => 'Usuario deletado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao deletar usuario: ' . $e->getMessage()]);
    }
}
?>