<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type");

// Conexao com o .env
$dotenv = parse_ini_file(__DIR__ . '../../.env');
$dbHost = $dotenv['DB_HOST'];
$dbUser = $dotenv['DB_USER'];
$dbPass = $dotenv['DB_PASSWORD'];
$dbName = $dotenv['DB_NAME'];

// Conexao com o banco de dados
try {
    $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha na conexao com o banco de dados: ' . $e->getMessage()]);
    exit;
}

// Roteamento
$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;

// Verifica se o script esta sendo acessado diretamente
if (basename($_SERVER['SCRIPT_FILENAME']) === 'status.php') {
    switch ($method) {
        case 'GET':
            if ($id) {
                getStatus_chamado($pdo, $id);
            } else {
                getAllStatus_chamados($pdo);
            }
            break;
        case 'POST':
            createStatus_chamado($pdo);
            break;
        case 'PUT':
            if ($id) {
                updateStatus_chamado($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do status_chamado nao especificado']);
            }
            break;
        case 'DELETE':
            if ($id) {
                deleteStatus_chamado($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do status_chamado nao especificado']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Metodo nao permitido']);
    }
} else {
    http_response_code(404);
    echo json_encode(['error' => 'Endpoint nao encontrado']);
}

// Funcoes CRUD

function getAllStatus_chamados($pdo) {
    try {
        $stmt = $pdo->query("SELECT * FROM status_chamado");
        $status_chamados = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($status_chamados);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar status_chamados: ' . $e->getMessage()]);
    }
}

function getStatus_chamado($pdo, $id) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM status_chamado WHERE id = ?");
        $stmt->execute([$id]);
        $status_chamado = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($status_chamado) {
            echo json_encode($status_chamado);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Status_chamado nao encontrado']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar status_chamado: ' . $e->getMessage()]);
    }
}

function createStatus_chamado($pdo) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados invalidos']);
        return;
    }
    
    if (!isset($data['descricao'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "descricao" faltando']);
        return;
    }
    if (strlen($data['descricao']) > 50) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "descricao" deve ter no maximo 50 caracteres']);
        return;
    }


    try {
        $sql = "INSERT INTO status_chamado (`descricao`) VALUES (:descricao)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':descricao' => $data['descricao']
        ]);
        
        $id = $pdo->lastInsertId();
        http_response_code(201);
        echo json_encode(['id' => $id, 'message' => 'Status_chamado criado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao criar status_chamado: ' . $e->getMessage()]);
    }
}

function updateStatus_chamado($pdo, $id) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados invalidos']);
        return;
    }
    
    if (!isset($data['descricao'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "descricao" faltando']);
        return;
    }
    if (strlen($data['descricao']) > 50) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "descricao" deve ter no maximo 50 caracteres']);
        return;
    }


    try {
        // Verifica se o status_chamado existe
        $check = $pdo->prepare("SELECT id FROM status_chamado WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Status_chamado nao encontrado']);
            return;
        }
        
        $sql = "UPDATE status_chamado SET `descricao` = :descricao WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':descricao' => $data['descricao'], ':id' => $id
        ]);
        
        echo json_encode(['message' => 'Status_chamado atualizado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao atualizar status_chamado: ' . $e->getMessage()]);
    }
}

function deleteStatus_chamado($pdo, $id) {
    try {
        // Verifica se o status_chamado existe
        $check = $pdo->prepare("SELECT id FROM status_chamado WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Status_chamado nao encontrado']);
            return;
        }
        
        $stmt = $pdo->prepare("DELETE FROM status_chamado WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode(['message' => 'Status_chamado deletado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao deletar status_chamado: ' . $e->getMessage()]);
    }
}
?>