<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type");

// Conexão com o .env
$dotenv = parse_ini_file(__DIR__ . '../../.env');
$dbHost = $dotenv['DB_HOST'];
$dbUser = $dotenv['DB_USER'];
$dbPass = $dotenv['DB_PASSWORD'];
$dbName = $dotenv['DB_NAME'];

// Conexão com o banco de dados
try {
    $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha na conexão com o banco de dados: ' . $e->getMessage()]);
    exit;
}

// Roteamento
$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;

// Verifica se o script está sendo acessado diretamente
if (basename($_SERVER['SCRIPT_FILENAME']) === 'eqp_tamanho.php') {
    switch ($method) {
        case 'GET':
            if ($id) {
                getTamanho($pdo, $id);
            } else {
                getAllTamanhos($pdo);
            }
            break;
        case 'POST':
            createTamanho($pdo);
            break;
        case 'PUT':
            if ($id) {
                updateTamanho($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do tamanho não especificado']);
            }
            break;
        case 'DELETE':
            if ($id) {
                deleteTamanho($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do tamanho não especificado']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Método não permitido']);
    }
} else {
    http_response_code(404);
    echo json_encode(['error' => 'Endpoint não encontrado']);
}

// Funções CRUD

function getAllTamanhos($pdo) {
    try {
        $stmt = $pdo->query("SELECT * FROM eq_tamanho");
        $tamanhos = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($tamanhos);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar tamanhos: ' . $e->getMessage()]);
    }
}

function getTamanho($pdo, $id) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM eq_tamanho WHERE id = ?");
        $stmt->execute([$id]);
        $tamanho = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($tamanho) {
            echo json_encode($tamanho);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Tamanho não encontrado']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar tamanho: ' . $e->getMessage()]);
    }
}

function createTamanho($pdo) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data || !isset($data['nome'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados inválidos ou campo "nome" faltando']);
        return;
    }
    
    // Validação adicional para o limite de caracteres (50)
    if (strlen($data['nome']) > 50) {
        http_response_code(400);
        echo json_encode(['error' => 'O nome do tamanho deve ter no máximo 50 caracteres']);
        return;
    }
    
    try {
        $sql = "INSERT INTO eq_tamanho (nome) VALUES (:nome)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([':nome' => $data['nome']]);
        
        $id = $pdo->lastInsertId();
        http_response_code(201);
        echo json_encode(['id' => $id, 'message' => 'Tamanho criado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao criar tamanho: ' . $e->getMessage()]);
    }
}

function updateTamanho($pdo, $id) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data || !isset($data['nome'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados inválidos ou campo "nome" faltando']);
        return;
    }
    
    // Validação adicional para o limite de caracteres (50)
    if (strlen($data['nome']) > 50) {
        http_response_code(400);
        echo json_encode(['error' => 'O nome do tamanho deve ter no máximo 50 caracteres']);
        return;
    }
    
    try {
        // Verifica se o tamanho existe
        $check = $pdo->prepare("SELECT id FROM eq_tamanho WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Tamanho não encontrado']);
            return;
        }
        
        $sql = "UPDATE eq_tamanho SET nome = :nome WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':id' => $id,
            ':nome' => $data['nome']
        ]);
        
        echo json_encode(['message' => 'Tamanho atualizado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao atualizar tamanho: ' . $e->getMessage()]);
    }
}

function deleteTamanho($pdo, $id) {
    try {
        // Verifica se o tamanho existe
        $check = $pdo->prepare("SELECT id FROM eq_tamanho WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Tamanho não encontrado']);
            return;
        }
        
        $stmt = $pdo->prepare("DELETE FROM eq_tamanho WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode(['message' => 'Tamanho deletado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao deletar tamanho: ' . $e->getMessage()]);
    }
}
?>