<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type");

// Conexão com o .env
$dotenv = parse_ini_file(__DIR__ . '../../.env');
$dbHost = $dotenv['DB_HOST'];
$dbUser = $dotenv['DB_USER'];
$dbPass = $dotenv['DB_PASSWORD'];
$dbName = $dotenv['DB_NAME'];

// Conexão com o banco de dados
try {
    $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha na conexão com o banco de dados: ' . $e->getMessage()]);
    exit;
}

// Roteamento
$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;

// Verifica se o script está sendo acessado diretamente
if (basename($_SERVER['SCRIPT_FILENAME']) === 'chamados.php') {
    switch ($method) {
        case 'GET':
            if ($id) {
                getChamado($pdo, $id);
            } else {
                getAllChamados($pdo);
            }
            break;
        case 'POST':
            createChamado($pdo);
            break;
        case 'PUT':
            if ($id) {
                updateChamado($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do chamado não especificado']);
            }
            break;
        case 'DELETE':
            if ($id) {
                deleteChamado($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do chamado não especificado']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Método não permitido']);
    }
} else {
    http_response_code(404);
    echo json_encode(['error' => 'Endpoint não encontrado']);
}

// Funções para manipulação dos chamados

function getAllChamados($pdo) {
    try {
        $query = "SELECT c.*, cl.nome_rs as cliente_nome, st.descricao as servico_tipo_desc, 
                 s.descricao as servico_desc, sol.solucao, se.descricao as atendente_nome
                 FROM chamado c
                 LEFT JOIN cliente cl ON c.fk_cliente = cl.id
                 LEFT JOIN servico_tipo st ON c.fk_servico_tipo = st.id
                 LEFT JOIN servico s ON c.fk_servico = s.id
                 LEFT JOIN solucao sol ON c.fk_solucao_proposta = sol.id
                 LEFT JOIN servico_especialistas se ON c.fk_atendente = se.id";
        
        $stmt = $pdo->query($query);
        $chamados = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($chamados);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar chamados: ' . $e->getMessage()]);
    }
}

function getChamado($pdo, $id) {
    try {
        $query = "SELECT c.*, cl.nome_rs as cliente_nome, st.descricao as servico_tipo_desc, 
                 s.descricao as servico_desc, sol.solucao, se.descricao as atendente_nome
                 FROM chamado c
                 LEFT JOIN cliente cl ON c.fk_cliente = cl.id
                 LEFT JOIN servico_tipo st ON c.fk_servico_tipo = st.id
                 LEFT JOIN servico s ON c.fk_servico = s.id
                 LEFT JOIN solucao sol ON c.fk_solucao_proposta = sol.id
                 LEFT JOIN servico_especialistas se ON c.fk_atendente = se.id
                 WHERE c.id = ?";
        
        $stmt = $pdo->prepare($query);
        $stmt->execute([$id]);
        $chamado = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($chamado) {
            echo json_encode($chamado);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Chamado não encontrado']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar chamado: ' . $e->getMessage()]);
    }
}

function createChamado($pdo) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    $requiredFields = ['fk_cliente', 'cliente_solicitante', 'serie', 'maquina', 'setor', 
                      'endereco', 'cidade', 'uf', 'fk_servico_tipo', 'fk_servico', 
                      'fk_atendente', 'detalhes'];
    
    foreach ($requiredFields as $field) {
        if (!isset($data[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Campo '$field' é obrigatório"]);
            return;
        }
    }
    
    try {
        // Primeiro, recupera o nome do cliente para preencher cliente_solicitante
        $stmtCliente = $pdo->prepare("SELECT nome_rs FROM cliente WHERE id = ?");
        $stmtCliente->execute([$data['fk_cliente']]);
        $cliente = $stmtCliente->fetch(PDO::FETCH_ASSOC);
        
        if (!$cliente) {
            http_response_code(404);
            echo json_encode(['error' => 'Cliente não encontrado']);
            return;
        }
        
        $data['cliente_solicitante'] = $cliente['nome_rs'];
        
        // Insere o chamado
        $sql = "INSERT INTO chamado (fk_cliente, cliente_solicitante, serie, maquina, setor, 
                endereco, cidade, uf, fk_servico_tipo, fk_servico, fk_solucao_proposta, 
                fk_atendente, detalhes, anexo)
                VALUES (:fk_cliente, :cliente_solicitante, :serie, :maquina, :setor, 
                :endereco, :cidade, :uf, :fk_servico_tipo, :fk_servico, :fk_solucao_proposta, 
                :fk_atendente, :detalhes, :anexo)";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':fk_cliente' => $data['fk_cliente'],
            ':cliente_solicitante' => $data['cliente_solicitante'],
            ':serie' => $data['serie'],
            ':maquina' => $data['maquina'],
            ':setor' => $data['setor'],
            ':endereco' => $data['endereco'],
            ':cidade' => $data['cidade'],
            ':uf' => $data['uf'],
            ':fk_servico_tipo' => $data['fk_servico_tipo'],
            ':fk_servico' => $data['fk_servico'],
            ':fk_solucao_proposta' => $data['fk_solucao_proposta'] ?? null,
            ':fk_atendente' => $data['fk_atendente'],
            ':detalhes' => $data['detalhes'],
            ':anexo' => $data['anexo'] ?? null
        ]);
        
        $id = $pdo->lastInsertId();
        http_response_code(201);
        echo json_encode(['id' => $id, 'message' => 'Chamado criado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao criar chamado: ' . $e->getMessage()]);
    }
}

function updateChamado($pdo, $id) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados inválidos']);
        return;
    }
    
    try {
        // Verifica se o chamado existe
        $check = $pdo->prepare("SELECT id FROM chamado WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Chamado não encontrado']);
            return;
        }
        
        // Atualiza o cliente_solicitante se fk_cliente foi alterado
        if (isset($data['fk_cliente'])) {
            $stmtCliente = $pdo->prepare("SELECT nome_rs FROM cliente WHERE id = ?");
            $stmtCliente->execute([$data['fk_cliente']]);
            $cliente = $stmtCliente->fetch(PDO::FETCH_ASSOC);
            
            if (!$cliente) {
                http_response_code(404);
                echo json_encode(['error' => 'Cliente não encontrado']);
                return;
            }
            
            $data['cliente_solicitante'] = $cliente['nome_rs'];
        }
        
        // Monta a query dinamicamente com os campos fornecidos
        $fields = [];
        $params = [':id' => $id];
        
        $allowedFields = ['fk_cliente', 'cliente_solicitante', 'serie', 'maquina', 'setor', 
                         'endereco', 'cidade', 'uf', 'fk_servico_tipo', 'fk_servico', 
                         'fk_solucao_proposta', 'fk_atendente', 'detalhes', 'anexo'];
        
        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $fields[] = "$field = :$field";
                $params[":$field"] = $data[$field];
            }
        }
        
        if (empty($fields)) {
            http_response_code(400);
            echo json_encode(['error' => 'Nenhum campo válido para atualização']);
            return;
        }
        
        $sql = "UPDATE chamado SET " . implode(', ', $fields) . " WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        
        echo json_encode(['message' => 'Chamado atualizado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao atualizar chamado: ' . $e->getMessage()]);
    }
}

function deleteChamado($pdo, $id) {
    try {
        // Verifica se o chamado existe
        $check = $pdo->prepare("SELECT id FROM chamado WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Chamado não encontrado']);
            return;
        }
        
        // Verifica se existem itens de atendimento associados
        $checkItens = $pdo->prepare("SELECT id FROM chamado_atendimento WHERE fk_chamado = ?");
        $checkItens->execute([$id]);
        
        if ($checkItens->fetch()) {
            http_response_code(400);
            echo json_encode(['error' => 'Não é possível deletar chamado com itens de atendimento associados']);
            return;
        }
        
        $stmt = $pdo->prepare("DELETE FROM chamado WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode(['message' => 'Chamado deletado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao deletar chamado: ' . $e->getMessage()]);
    }
}