<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type");

// Conexao com o .env
$dotenv = parse_ini_file(__DIR__ . '../../.env');
$dbHost = $dotenv['DB_HOST'];
$dbUser = $dotenv['DB_USER'];
$dbPass = $dotenv['DB_PASSWORD'];
$dbName = $dotenv['DB_NAME'];

// Conexao com o banco de dados
try {
    $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha na conexao com o banco de dados: ' . $e->getMessage()]);
    exit;
}

// Roteamento
$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;

// Verifica se o script esta sendo acessado diretamente
if (basename($_SERVER['SCRIPT_FILENAME']) === 'produto_saida.php') {
    switch ($method) {
        case 'GET':
            if ($id) {
                getProduto_saida($pdo, $id);
            } else {
                getAllProduto_saidas($pdo);
            }
            break;
        case 'POST':
            createProduto_saida($pdo);
            break;
        case 'PUT':
            if ($id) {
                updateProduto_saida($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do produto_saida nao especificado']);
            }
            break;
        case 'DELETE':
            if ($id) {
                deleteProduto_saida($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do produto_saida nao especificado']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Metodo nao permitido']);
    }
} else {
    http_response_code(404);
    echo json_encode(['error' => 'Endpoint nao encontrado']);
}

// Funcoes CRUD

function getAllProduto_saidas($pdo) {
    try {
        $stmt = $pdo->query("SELECT * FROM produto_saida");
        $produto_saidas = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($produto_saidas);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar produto_saidas: ' . $e->getMessage()]);
    }
}

function getProduto_saida($pdo, $id) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM produto_saida WHERE id = ?");
        $stmt->execute([$id]);
        $produto_saida = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($produto_saida) {
            echo json_encode($produto_saida);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Produto_saida nao encontrado']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar produto_saida: ' . $e->getMessage()]);
    }
}

function createProduto_saida($pdo) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados invalidos']);
        return;
    }
    
    if (!isset($data['fk_chamado_acomp'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "fk_chamado_acomp" faltando']);
        return;
    }

    if (!isset($data['produto'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "produto" faltando']);
        return;
    }
    if (strlen($data['produto']) > 100) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "produto" deve ter no maximo 100 caracteres']);
        return;
    }

    if (!isset($data['qtd'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "qtd" faltando']);
        return;
    }

    if (!isset($data['solicitante'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "solicitante" faltando']);
        return;
    }
    if (strlen($data['solicitante']) > 300) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "solicitante" deve ter no maximo 300 caracteres']);
        return;
    }


    try {
        $sql = "INSERT INTO produto_saida (`fk_chamado_acomp`, `produto`, `qtd`, `solicitante`) VALUES (:fk_chamado_acomp, :produto, :qtd, :solicitante)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':fk_chamado_acomp' => $data['fk_chamado_acomp'], ':produto' => $data['produto'], ':qtd' => $data['qtd'], ':solicitante' => $data['solicitante']
        ]);
        
        $id = $pdo->lastInsertId();
        http_response_code(201);
        echo json_encode(['id' => $id, 'message' => 'Produto_saida criado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao criar produto_saida: ' . $e->getMessage()]);
    }
}

function updateProduto_saida($pdo, $id) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados invalidos']);
        return;
    }
    
    if (!isset($data['fk_chamado_acomp'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "fk_chamado_acomp" faltando']);
        return;
    }

    if (!isset($data['produto'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "produto" faltando']);
        return;
    }
    if (strlen($data['produto']) > 100) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "produto" deve ter no maximo 100 caracteres']);
        return;
    }

    if (!isset($data['qtd'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "qtd" faltando']);
        return;
    }

    if (!isset($data['solicitante'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "solicitante" faltando']);
        return;
    }
    if (strlen($data['solicitante']) > 300) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "solicitante" deve ter no maximo 300 caracteres']);
        return;
    }


    try {
        // Verifica se o produto_saida existe
        $check = $pdo->prepare("SELECT id FROM produto_saida WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Produto_saida nao encontrado']);
            return;
        }
        
        $sql = "UPDATE produto_saida SET `fk_chamado_acomp` = :fk_chamado_acomp, `produto` = :produto, `qtd` = :qtd, `solicitante` = :solicitante WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':fk_chamado_acomp' => $data['fk_chamado_acomp'], ':produto' => $data['produto'], ':qtd' => $data['qtd'], ':solicitante' => $data['solicitante'], ':id' => $id
        ]);
        
        echo json_encode(['message' => 'Produto_saida atualizado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao atualizar produto_saida: ' . $e->getMessage()]);
    }
}

function deleteProduto_saida($pdo, $id) {
    try {
        // Verifica se o produto_saida existe
        $check = $pdo->prepare("SELECT id FROM produto_saida WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Produto_saida nao encontrado']);
            return;
        }
        
        $stmt = $pdo->prepare("DELETE FROM produto_saida WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode(['message' => 'Produto_saida deletado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao deletar produto_saida: ' . $e->getMessage()]);
    }
}
?>